import { NextResponse } from "next/server";
import { createUser, listUsers } from "@/lib/userRepository";

type Body = {
  name?: string;
  email?: string;
  role?: string;
};

function validateUserPayload(body: Body) {
  const errors: string[] = [];

  if (!body.name?.trim()) {
    errors.push("名前は必須です。");
  }

  if (!body.email?.trim()) {
    errors.push("メールアドレスは必須です。");
  } else {
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailRegex.test(body.email)) {
      errors.push("メールアドレスの形式が正しくありません。");
    }
  }

  if (!body.role?.trim()) {
    errors.push("役割は必須です。");
  }

  return errors;
}

export async function GET() {
  try {
    const users = await listUsers();
    return NextResponse.json(users);
  } catch (error) {
    console.error(error);
    return NextResponse.json(
      { message: "ユーザー一覧の取得に失敗しました。" },
      { status: 500 }
    );
  }
}

export async function POST(request: Request) {
  try {
    const body = (await request.json()) as Body;
    const errors = validateUserPayload(body);

    if (errors.length) {
      return NextResponse.json({ message: errors.join("\n") }, { status: 400 });
    }

    const user = await createUser({
      name: body.name!.trim(),
      email: body.email!.trim(),
      role: body.role!.trim(),
    });

    return NextResponse.json(user, { status: 201 });
  } catch (error) {
    console.error(error);
    const message =
      error instanceof Error && error.message.includes("ER_DUP_ENTRY")
        ? "指定されたメールアドレスは既に使用されています。"
        : "ユーザーの作成に失敗しました。";

    return NextResponse.json({ message }, { status: 500 });
  }
}
