"use client";

import { FormEvent, useCallback, useEffect, useMemo, useState } from "react";
import styles from "./page.module.css";

type User = {
  id: number;
  name: string;
  email: string;
  role: string;
  createdAt: string;
};

type ModalState =
  | { type: "create" }
  | { type: "edit"; user: User }
  | { type: "delete"; user: User };

type FormState = {
  name: string;
  email: string;
  role: string;
};

const emptyFormState: FormState = {
  name: "",
  email: "",
  role: "",
};

export default function Home() {
  const [sidebarOpen, setSidebarOpen] = useState(true);
  const [activeMenu, setActiveMenu] = useState<"users" | "projects">("users");
  const [users, setUsers] = useState<User[]>([]);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [modal, setModal] = useState<ModalState | null>(null);
  const [formData, setFormData] = useState<FormState>(emptyFormState);
  const [formError, setFormError] = useState<string | null>(null);
  const [saving, setSaving] = useState(false);

  const loadUsers = useCallback(async () => {
    setLoading(true);
    setError(null);

    try {
      const response = await fetch("/api/users", { cache: "no-store" });

      if (!response.ok) {
        const body = await response.json().catch(() => null);
        const message =
          typeof body?.message === "string"
            ? body.message
            : "ユーザーの取得に失敗しました。";
        throw new Error(message);
      }

      const body = (await response.json()) as User[];
      setUsers(body);
    } catch (err) {
      const message =
        err instanceof Error
          ? err.message
          : "ユーザーの取得に失敗しました。";
      setError(message);
    } finally {
      setLoading(false);
    }
  }, []);

  useEffect(() => {
    if (activeMenu === "users") {
      loadUsers();
    }
  }, [activeMenu, loadUsers]);

  const toggleSidebar = () => setSidebarOpen((prev) => !prev);

  const openCreateModal = () => {
    setFormData(emptyFormState);
    setFormError(null);
    setModal({ type: "create" });
  };

  const openEditModal = (user: User) => {
    setFormData({
      name: user.name,
      email: user.email,
      role: user.role,
    });
    setFormError(null);
    setModal({ type: "edit", user });
  };

  const openDeleteModal = (user: User) => {
    setFormError(null);
    setModal({ type: "delete", user });
  };

  const closeModal = () => {
    if (saving) {
      return;
    }
    setModal(null);
  };

  const handleFormChange =
    (field: keyof FormState) => (event: FormEvent<HTMLInputElement>) => {
      const target = event.currentTarget;
      setFormData((prev) => ({
        ...prev,
        [field]: target.value,
      }));
    };

  const submitForm = async () => {
    if (!modal || modal.type === "delete") {
      return;
    }

    setSaving(true);
    setFormError(null);

    try {
      const payload = {
        name: formData.name.trim(),
        email: formData.email.trim(),
        role: formData.role.trim(),
      };

      const response = await fetch(
        modal.type === "create" ? "/api/users" : `/api/users/${modal.user.id}`,
        {
          method: modal.type === "create" ? "POST" : "PUT",
          headers: {
            "Content-Type": "application/json",
          },
          body: JSON.stringify(payload),
        }
      );

      if (!response.ok) {
        const body = await response.json().catch(() => null);
        const message =
          typeof body?.message === "string"
            ? body.message
            : "ユーザーの保存に失敗しました。";
        throw new Error(message);
      }

      await loadUsers();
      setModal(null);
    } catch (err) {
      const message =
        err instanceof Error
          ? err.message
          : "ユーザーの保存に失敗しました。";
      setFormError(message);
    } finally {
      setSaving(false);
    }
  };

  const confirmDelete = async () => {
    if (!modal || modal.type !== "delete") {
      return;
    }

    setSaving(true);
    setFormError(null);

    try {
      const response = await fetch(`/api/users/${modal.user.id}`, {
        method: "DELETE",
      });

      if (!response.ok) {
        const body = await response.json().catch(() => null);
        const message =
          typeof body?.message === "string"
            ? body.message
            : "ユーザーの削除に失敗しました。";
        throw new Error(message);
      }

      await loadUsers();
      setModal(null);
    } catch (err) {
      const message =
        err instanceof Error
          ? err.message
          : "ユーザーの削除に失敗しました。";
      setFormError(message);
    } finally {
      setSaving(false);
    }
  };

  const usersView = useMemo(() => {
    if (loading) {
      return <div className={styles.status}>読み込み中です...</div>;
    }

    if (error) {
      return <div className={styles.errorBox}>{error}</div>;
    }

    if (!users.length) {
      return (
        <div className={styles.placeholder}>
          登録されているユーザーはまだありません。
        </div>
      );
    }

    return (
      <table className={styles.usersTable}>
        <thead>
          <tr>
            <th>名前</th>
            <th>メールアドレス</th>
            <th>役割</th>
            <th>作成日</th>
            <th>操作</th>
          </tr>
        </thead>
        <tbody>
          {users.map((user) => (
            <tr key={user.id}>
              <td>{user.name}</td>
              <td>{user.email}</td>
              <td>{user.role}</td>
              <td>
                {new Intl.DateTimeFormat("ja-JP", {
                  year: "numeric",
                  month: "2-digit",
                  day: "2-digit",
                  hour: "2-digit",
                  minute: "2-digit",
                }).format(new Date(user.createdAt))}
              </td>
              <td>
                <div className={styles.actionButtons}>
                  <button
                    className={styles.secondaryButton}
                    type="button"
                    onClick={() => openEditModal(user)}
                  >
                    更新
                  </button>
                  <button
                    className={styles.dangerButton}
                    type="button"
                    onClick={() => openDeleteModal(user)}
                  >
                    削除
                  </button>
                </div>
              </td>
            </tr>
          ))}
        </tbody>
      </table>
    );
  }, [error, loading, users]);

  const mainContent =
    activeMenu === "users" ? (
      <>
        <div className={styles.mainHeader}>
          <div>
            <h2 className={styles.sectionTitle}>ユーザー一覧</h2>
            <p className={styles.sectionSubtitle}>
              登録済みのユーザーを確認・管理できます。
            </p>
          </div>
          <button
            className={styles.primaryButton}
            type="button"
            onClick={openCreateModal}
          >
            新規作成
          </button>
        </div>
        {usersView}
      </>
    ) : (
      <div className={styles.placeholder}>
        プロジェクト機能は現在準備中です。
      </div>
    );

  return (
    <div className={styles.app}>
      <header className={styles.header}>
        <button
          className={`${styles.hamburger} ${
            sidebarOpen ? styles.hamburgerOpen : ""
          }`}
          type="button"
          onClick={toggleSidebar}
          aria-label="メニューを開閉"
          aria-expanded={sidebarOpen}
        >
          <span />
          <span />
          <span />
        </button>
        <h1 className={styles.title}>BizApp 管理コンソール</h1>
      </header>
      <div className={styles.body}>
        <nav
          className={`${styles.sidebar} ${
            sidebarOpen ? styles.sidebarOpen : styles.sidebarClosed
          }`}
          aria-hidden={!sidebarOpen}
        >
          <button
            className={`${styles.navItem} ${
              activeMenu === "users" ? styles.navItemActive : ""
            }`}
            type="button"
            onClick={() => setActiveMenu("users")}
          >
            ユーザー
          </button>
          <button
            className={`${styles.navItem} ${
              activeMenu === "projects" ? styles.navItemActive : ""
            }`}
            type="button"
            onClick={() => setActiveMenu("projects")}
          >
            プロジェクト
          </button>
        </nav>
        <main className={styles.mainContent}>{mainContent}</main>
      </div>

      {modal && (
        <div className={styles.modalOverlay} role="dialog" aria-modal="true">
          <div className={styles.modal}>
            {modal.type === "delete" ? (
              <>
                <h2 className={styles.modalTitle}>ユーザー削除</h2>
                <p className={styles.modalBody}>
                  {`${
                    modal.user.name
                  } さんを削除しますか？この操作は取り消せません。`}
                </p>
                {formError && (
                  <div className={styles.errorBox}>{formError}</div>
                )}
                <div className={styles.modalActions}>
                  <button
                    className={styles.secondaryButton}
                    type="button"
                    onClick={closeModal}
                    disabled={saving}
                  >
                    キャンセル
                  </button>
                  <button
                    className={styles.dangerButton}
                    type="button"
                    onClick={confirmDelete}
                    disabled={saving}
                  >
                    {saving ? "削除中..." : "削除する"}
                  </button>
                </div>
              </>
            ) : (
              <>
                <h2 className={styles.modalTitle}>
                  {modal.type === "create" ? "ユーザー新規作成" : "ユーザー更新"}
                </h2>
                <div className={styles.modalBody}>
                  <div className={styles.inputGroup}>
                    <label htmlFor="name">名前</label>
                    <input
                      id="name"
                      value={formData.name}
                      onChange={handleFormChange("name")}
                      placeholder="山田 太郎"
                    />
                  </div>
                  <div className={styles.inputGroup}>
                    <label htmlFor="email">メールアドレス</label>
                    <input
                      id="email"
                      type="email"
                      value={formData.email}
                      onChange={handleFormChange("email")}
                      placeholder="taro@example.com"
                    />
                  </div>
                  <div className={styles.inputGroup}>
                    <label htmlFor="role">役割</label>
                    <input
                      id="role"
                      value={formData.role}
                      onChange={handleFormChange("role")}
                      placeholder="管理者"
                    />
                  </div>
                </div>
                {formError && (
                  <div className={styles.errorBox}>{formError}</div>
                )}
                <div className={styles.modalActions}>
                  <button
                    className={styles.secondaryButton}
                    type="button"
                    onClick={closeModal}
                    disabled={saving}
                  >
                    キャンセル
                  </button>
                  <button
                    className={styles.primaryButton}
                    type="button"
                    onClick={submitForm}
                    disabled={saving}
                  >
                    {saving ? "保存中..." : "保存する"}
                  </button>
                </div>
              </>
            )}
          </div>
        </div>
      )}
    </div>
  );
}
