import { ResultSetHeader, RowDataPacket } from "mysql2";
import { getPool } from "./db";

export type DbUser = {
  id: number;
  name: string;
  email: string;
  role: string;
  createdAt: string;
  updatedAt: string;
};

async function ensureUsersTable(): Promise<void> {
  const pool = getPool();
  await pool.query(`
    CREATE TABLE IF NOT EXISTS users (
      id INT NOT NULL AUTO_INCREMENT,
      name VARCHAR(100) NOT NULL,
      email VARCHAR(255) NOT NULL UNIQUE,
      role VARCHAR(100) NOT NULL,
      created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
      PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
  `);
}

function mapRowToUser(row: RowDataPacket): DbUser {
  return {
    id: row.id,
    name: row.name,
    email: row.email,
    role: row.role,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

export async function listUsers(): Promise<DbUser[]> {
  await ensureUsersTable();
  const pool = getPool();
  const [rows] = await pool.query<RowDataPacket[]>(
    "SELECT id, name, email, role, created_at, updated_at FROM users ORDER BY created_at DESC"
  );

  return rows.map(mapRowToUser);
}

export async function createUser(input: {
  name: string;
  email: string;
  role: string;
}): Promise<DbUser> {
  await ensureUsersTable();
  const pool = getPool();
  const [result] = await pool.execute<ResultSetHeader>(
    "INSERT INTO users (name, email, role) VALUES (?, ?, ?)",
    [input.name, input.email, input.role]
  );

  const insertedId = result.insertId;

  return getUserById(insertedId);
}

export async function getUserById(id: number): Promise<DbUser> {
  const pool = getPool();
  const [rows] = await pool.query<RowDataPacket[]>(
    "SELECT id, name, email, role, created_at, updated_at FROM users WHERE id = ?",
    [id]
  );

  if (!rows.length) {
    throw new Error("USER_NOT_FOUND");
  }

  return mapRowToUser(rows[0]);
}

export async function updateUser(
  id: number,
  input: { name: string; email: string; role: string }
): Promise<DbUser> {
  await ensureUsersTable();
  const pool = getPool();
  const [result] = await pool.execute<ResultSetHeader>(
    "UPDATE users SET name = ?, email = ?, role = ? WHERE id = ?",
    [input.name, input.email, input.role, id]
  );

  if (result.affectedRows === 0) {
    throw new Error("USER_NOT_FOUND");
  }

  return getUserById(id);
}

export async function deleteUser(id: number): Promise<void> {
  await ensureUsersTable();
  const pool = getPool();
  const [result] = await pool.execute<ResultSetHeader>(
    "DELETE FROM users WHERE id = ?",
    [id]
  );

  if (result.affectedRows === 0) {
    throw new Error("USER_NOT_FOUND");
  }
}
